package com.polycom.sampleapps.servlet.common;

import java.io.File;
import java.io.IOException;
import java.util.Enumeration;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.StringTokenizer;
import java.util.Vector;
import java.util.logging.FileHandler;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.logging.SimpleFormatter;

import javax.servlet.RequestDispatcher;
import javax.servlet.ServletContext;
import javax.servlet.ServletException;
import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.apache.http.HttpRequest;

import com.polycom.sampleapps.servlet.common.data.Application;
import com.polycom.sampleapps.servlet.common.data.Item;
import com.polycom.sampleapps.servlet.common.data.User;
import com.polycom.sampleapps.servlet.common.data.UserProfile;
import com.polycom.sampleapps.servlet.common.parsers.UserXmlParser;
import com.polycom.sampleapps.servlet.common.parsers.UsersXmlParser;
import com.polycom.sampleapps.servlet.common.persistance.PersistUser;
import com.polycom.sampleapps.servlet.common.persistance.PersistUsers;
import com.polycom.sampleapps.servlet.exception.UserAlreadyExistsException;
import com.polycom.sampleapps.servlet.news.NewsManager;
import com.polycom.sampleapps.servlet.news.NewsRefreshManager;
import com.polycom.sampleapps.servlet.stocks.StockManager;
import com.polycom.sampleapps.servlet.stocks.StockQuoteRefreshManager;
import com.polycom.sampleapps.servlet.thought.ThoughtForTheDayManager;
import com.polycom.sampleapps.servlet.thought.ThoughtRefreshManager;
import com.polycom.sampleapps.servlet.weather.WeatherManager;
import com.polycom.sampleapps.servlet.weather.WeatherRefreshManager;

/**
 *
 * <p>Title: ApplicationContext</p>
 *
 * <p>Description: Caches Applications, users and user specific information defined in XML files.
 * Provides utility methods to get the relavant information from cache. </p>
 *
 * <p>Copyright:* Copyright (c) 2006 Polycom Canada Ltd.</p>
 *
 * <p>Company: Polycom, Inc</p>
 *
 * @author T Ravikanth
 * @version 1.0
 */
public final class ApplicationContext
{
	public static final String XML_DIR = "xml";
	
	public static final long UNUSED_PERIOD=1000*60*5;

	private UserProfile m_defaultUser; //
	private Vector<User> m_users;
	private String m_appRelativePath;
	private Hashtable<String, UserProfile> m_systemUsers;
	// MAINTAINING TEMPORARY CACHE
	private Hashtable<String, UserProfile> m_loggedInUsersCache;
	private static ApplicationContext m_self;
	private RefreshManager m_stockRefreshManager;
	private RefreshManager m_weatherRefreshManager;
	private RefreshManager m_newsRefreshManager;
	private ThoughtForTheDayManager m_thoughtRefreshManager;
	private static String m_appPath;
	private StockManager stockManager;

	private NewsManager newsManager;
	
	private WeatherManager weatherManager;

	// INITIALIZING LOGGER
	public static Logger logger = Logger.getLogger(ApplicationContext.class.
		getName());

	/**
	 * Initialized when LoginServlet/IdleServlet are invoked by the Idle/Main Browsers.
	 * This is responsible to initialize default user (ADMIN) and users from xml files placed under "xml" directory.
	 * @param a_appPath String
	 */
	private ApplicationContext(String a_appPath)
	{		m_appRelativePath = a_appPath;
		m_systemUsers = new Hashtable<String, UserProfile> ();
		m_loggedInUsersCache = new Hashtable<String, UserProfile> ();
		addLogHandler();
		stockManager = new StockManager(this);
		newsManager=new NewsManager(this);
		weatherManager=new WeatherManager(this);
		loadUsers();
		loadDefaultUser();
		createDirectories();
		BlockedThreadInterruptor.getInstance(this);
	}

	/**
	 * Initializes ApplicationContext for one time.
	 * @param a_context ServletContext
	 * @return ApplicationContext
	 */
	public static synchronized ApplicationContext getInstance(ServletContext
		a_context)
	{
		if (m_self == null)
		{
			m_self = new ApplicationContext(a_context.getRealPath(File.
				separator));
		}
		return m_self;
	}
	
	public static void test()
	{
		System.out.println("dkjfdjkhfkjhdfkhd");
	}

	/**
	 * Return Stock refresh manager.
	 * @return RefreshManager
	 */
	public RefreshManager getStockRefreshManager()
	{
		if (m_stockRefreshManager == null)
		{
			m_stockRefreshManager = StockQuoteRefreshManager.getInstance(this);
		}
		return m_stockRefreshManager;
	}
	
	public StockManager getStockManager()
	{
		return stockManager;
	}
	
	public NewsManager getNewsManager()
	{
		return newsManager;
	}
	
	public WeatherManager getWeatherManager()
	{
		return weatherManager;
	}
	

	/**
	 * Returns Weather refresh manger
	 * @return RefreshManager
	 */
	public RefreshManager getWeatherRefreshManager()
	{
		if (m_weatherRefreshManager == null)
		{
			m_weatherRefreshManager = WeatherRefreshManager.getInstance();
		}
		return m_weatherRefreshManager;
	}

	/**
	 * Returns News refresh manager
	 * @return RefreshManager
	 */
	public RefreshManager getNewsRefreshManager()
	{
		if (m_newsRefreshManager == null)
		{
			m_newsRefreshManager = NewsRefreshManager.getInstance();
		}
		return m_newsRefreshManager;
	}

	/**
	 * Returns Thought of the day refresh manager
	 * @return RefreshManager
	 */
	public ThoughtForTheDayManager getThoughtRefreshManager()
	{
		if (m_thoughtRefreshManager == null)
		{
			m_thoughtRefreshManager = ThoughtForTheDayManager.getInstannce();
		}
		return m_thoughtRefreshManager;
	}

	/**
	 * Checks the given username with existing users.
	 * @param a_userName String
	 * @return boolean
	 */
	public boolean isUserExists(String a_userName)
	{
		boolean result = false;
		for (User user : getUsers())
		{
			if (user.getName().equals(a_userName))
			{
				result = true;
				break;
			}
		}
		return result;
	}

	/**
	 * Will be used for simulator for loading the users profiles
	 * @param a_request
	 */
	public void addUser(HttpServletRequest a_request)
	{
		Cookie[] cookies = a_request.getCookies();
		if(cookies!=null&&cookies.length==4)
		{
			String userName=null,password=null,type=null;
			boolean adduser=false;
			for(Cookie cookie:cookies)
			{
				if(cookie.getName().equals("username"))
				{
					userName=cookie.getValue();
				}
				if(cookie.getName().equals("password"))
				{
					password=cookie.getValue();
				}
				if(cookie.getName().equals("usertype"))
				{
					type=cookie.getValue();
				}				
				if(cookie.getName().equals("simulator"))
				{
					adduser=true;
				}
			}
			
			if(userName!=null&&password!=null&&type!=null&&adduser)
			{	try {
					addUser(userName,password,type,true);
				} catch (UserAlreadyExistsException e) {
					e.printStackTrace();
				} catch (IOException e) {
					e.printStackTrace();
				}
			}
		}		
	}
	
	
	/**
	 * Checks with existing uses and if does not exist then adds the user and persist the user into the user.xml file with default applications
	 * @param a_userName String
	 * @param a_password String
	 * @param a_type String
	 * @throws UserAlreadyExistsException
	 */
	public void addUser(String a_userName, String a_password,
						String a_type,boolean isSimulator) throws
		UserAlreadyExistsException, IOException
	{
		
		if(isSimulator)
		{
			User user = new User(a_userName, a_password, a_type);
			UserProfile userProfile = new UserProfile();
			userProfile.setApplications(getDefaultEnabledApplications());
			//Check for simulator
			if(!m_users.contains(user))
				m_users.add(user);
			String usersFilePath = getXmlDirPath() +
				Constants.USERS_XML_FILE;
			PersistUsers.persist(m_users, usersFilePath);
			String userFilePath = getXmlDirPath() + user.getName() + ".xml";
			
			//IF xml file exists then register the user
			File userXMLFile=new File(userFilePath);
			if(userXMLFile.exists())
			{
				UserXmlParser parser = new UserXmlParser();
				UserProfile result = parser.parseUserXML(userFilePath);
				result.setUser(user);
				registerApplicationItemsWithRefreshManager(result);
			}
		}
		else{
				if (isUserExists(a_userName))
					{
						throw new UserAlreadyExistsException("User '" + a_userName +
														 "' already exists.");
					}
				else
					{
						User user = new User(a_userName, a_password, a_type);
						UserProfile userProfile = new UserProfile();
						userProfile.setApplications(getDefaultEnabledApplications());
						//Check for simulator
						if(!m_users.contains(user))
							m_users.add(user);
							try
							{
								String usersFilePath = getXmlDirPath() +
									Constants.USERS_XML_FILE;
								PersistUsers.persist(m_users, usersFilePath);
								String userFilePath = getXmlDirPath() + user.getName() + ".xml";
								
								//IF xml file exists then register the user
								File userXMLFile=new File(userFilePath);
								if(userXMLFile.exists())
								{
									UserXmlParser parser = new UserXmlParser();
									UserProfile result = parser.parseUserXML(userFilePath);
									result.setUser(user);
									registerApplicationItemsWithRefreshManager(result);
									//System.out.println("Registered user:"+a_userName);
								}
								else
								{//Create user.xml file 
									PersistUser.persist(userProfile, userFilePath);
								}
							}
							catch (IOException ex)
							{
								throw ex;
							}
					}
			}
	}

	/**
	 * This will return the "xml" directory path and used by the xml parser classes.
	 * @return String
	 */
	public String getXmlDirPath()
	{
		return m_appRelativePath + XML_DIR + File.separator;
	}

	/**
	 * Returns stock bmp files storage path
	 * @return String
	 */
	public String getStocksDirPath()
	{
		String result = getApplicationPath() + Constants.STOCKS_BMP_PATH;
		return result;
	}

	/**
	 * Returns Weather location bmp files storage path
	 * @return String
	 */
	public String getLocationsDirPath()
	{
		String result = getApplicationPath() + Constants.WEATHER_BMP_PATH;
		return result;
	}

	/**
	 * This method adds the username cookie to the response.
	 * If the user is end user, this will add the permanent cookie to the response.
	 * @param a_response HttpServletResponse
	 * @param a_user User
	 * @param a_permanent boolean
	 */
	public void addUsernameCookieToTheResponse(HttpServletResponse a_response,
											   User a_user, boolean a_permanent)
	{
		Cookie cookie = new Cookie(Constants.COOKIE_USERNAME, a_user.getName());
		if (a_permanent)
		{
			cookie.setMaxAge(100 * 365 * 24 * 60 * 60 * 1000);
		}
		a_response.addCookie(cookie);
	}

	/**
	 * Retrieves the username from cookie and returns the UserProfile for specified username.
	 * @param a_request HttpServletRequest
	 * @return UserProfile
	 */
	public UserProfile getUserFromCookie(HttpServletRequest
										 a_request)
	{
		//ApplicationContext.logger.log(Level.INFO,"entered to getUserFromCookie");
		UserProfile result = null;
		Cookie[] cookies = a_request.getCookies();
		/////////TEMPORARY CODE TO WORK WITH SIMULATOR///////////////
		/*Enumeration enum1 = a_request.getHeaderNames();
		   int count = 0;
		   while (enum1.hasMoreElements())
		   {
		 String ele = (String) enum1.nextElement();
		 String val = a_request.getHeader(ele);
		 //System.out.println(ele + " :   : " + val);
		 if ( (ele.startsWith("Set-Cookie") ||
			ele.startsWith("set-cookie")) &&
		  val.indexOf(Constants.COOKIE_USERNAME) != -1)
		 {
		  //System.out.println("User Name : " + val);
		  if (cookies == null)
		  {
		   cookies = new Cookie[1];
		  }
		  StringTokenizer stk = new StringTokenizer(val, "=");
		  cookies[count] = new Cookie(stk.nextToken(), stk.nextToken());
		  count++;
		 }
		   }*/
		/////////END TEMPORARY CODE TO WORK WITH SIMULATOR///////////////
		
		if (cookies != null)
		{
			for (Cookie cookie : cookies)
			{
				//System.out.println("userName in COOKIE : " + cookie.getName());
				if (cookie.getName().equals(Constants.COOKIE_USERNAME))
				{
					String userName = cookie.getValue();					
					result = getUser(userName, a_request.getRemoteAddr());
					break;
				}
			}
			if (result == null)
			{
				// GET USERPROFILE USING REMOTE ADDRESS
				result = getUser("", a_request.getRemoteAddr());
			}
						
		}
		//System.out.println("In:"+result);
		return result;
	}

	/**
	 * Registers logged in users configured application items with respective managers.
	 * @param a_userProfile UserProfile
	 */
	public void registerApplicationItemsWithRefreshManager(UserProfile
		a_userProfile)
	{
		if (a_userProfile != null)
		{
			if (a_userProfile.getName() != null)
			{
				m_systemUsers.put(a_userProfile.getName(), a_userProfile);
			}
			Application stockApp = a_userProfile.getStockApplication();
			if (stockApp != null)
			{
				Vector<Item> items = stockApp.getItems();	
				for (Item item : items)
				{
					stockManager.add(item.getName());
					//System.out.println(item.getName());
				}
			}
			Application weatherApp = a_userProfile.getWeatherApplication();
			if (weatherApp != null)
			{
				Vector<Item> items = weatherApp.getItems();
				for (Item item : items)
				{
					weatherManager.add(item.getName());
					//getWeatherRefreshManager().register(item.getName());
				}
			}
			Application newsApp = a_userProfile.getNewsApplication();
			if (newsApp != null)
			{
				Vector<Item> items = newsApp.getItems();
				for (Item item : items)
				{
					newsManager.add(item.getContentProviderUrl());
					//getNewsRefreshManager().register(item.getContentProviderUrl());
				}
			}
		}
	}

	/**
	 * Closes all the thread pools created by the managers.
	 */
	public void shutdownRefreshManagers()
	{
		m_stockRefreshManager.shutdown();
		m_weatherRefreshManager.shutdown();
		m_newsRefreshManager.shutdown();
		//m_thoughtRefreshManager.shutdown();
	}

	/**
	 * This will return the "images" directory path and used by the bmp converter to store the bmp file.
	 * @return String
	 */
	public String getApplicationPath()
	{
		return m_appRelativePath;
	}

	/**
	 * This methos is invoked by the LogoutServlet to remove the last logged in user from session.
	 * @param a_userName String
	 */
	public void logoutUser(String a_userName, String a_remoteAddr)
	{
		UserProfile user = getUser(a_userName, a_remoteAddr);
		if (user != null)
		{
			// PERSIST USER
			try
			{
				if (!user.getType().equals(Constants.IT_USER_TYPE))
				{
					String userFilePath = getXmlDirPath() + user.getName() +
						".xml";
					PersistUser.persist(user, userFilePath);
				}
			}
			catch (IOException ex)
			{
				logger.log(Level.SEVERE, ex.getMessage(), ex);
			}
		}
	}

	/**
	 * This method checks the given username is available.
	 * @param a_userName String
	 * @return User
	 */

	public UserProfile getUser(String a_userName,
							   String a_remoteAddr)
	{
		//ApplicationContext.logger.log(Level.INFO, "entered in getUser");
		UserProfile result = null;		
		if (m_systemUsers.containsKey(a_userName))
		{
			result = m_systemUsers.get(a_userName);
			if(a_remoteAddr!=null)
				m_loggedInUsersCache.put(a_remoteAddr, result);
			ApplicationContext.logger.log(Level.INFO,"contained in SystemUsers*************** " + result);				
		}
		else if (m_loggedInUsersCache.containsKey(a_remoteAddr))
		{
			result = m_loggedInUsersCache.get(a_remoteAddr);
			ApplicationContext.logger.log(Level.INFO,"adding to systemUsers --------------" + result);
			m_systemUsers.put(result.getName(), result);						
		}
		else
		{
			
			// READING USER SPECIFIC XML FILE
			if (a_userName != null && a_userName.length() > 0)
			{
				String userXmlFile = getXmlDirPath() + a_userName + ".xml";
				File file = new File(userXmlFile);				
				if (file.exists())
				{
					UserXmlParser parser = new UserXmlParser();
					result = parser.parseUserXML(userXmlFile);
					User user = getSystemUser(a_userName);
					result.setUser(user);
					registerApplicationItemsWithRefreshManager(result);
					ApplicationContext.logger.log(Level.INFO,"adding to loggedInUsersCache +++++++++++++ " + result);
					m_loggedInUsersCache.put(a_remoteAddr, result);		
				}
				if (result == null)
				{
					// CHECK FOR ADMIN
					User admin = getSystemUser(a_userName);
					ApplicationContext.logger.log(Level.INFO,"getSystemUser(a_userName) =========== " + admin);
					if (admin != null)
					{
						m_defaultUser.setUser(admin);
						m_loggedInUsersCache.put(a_remoteAddr, m_defaultUser);
						result = m_defaultUser;
					}
				}
			}
		}
		return result;
	}
	
	public boolean isSimulatorRequest(HttpServletRequest servletRequest)
	{
		Cookie[] cookies = servletRequest.getCookies();
		for(Cookie cookie:cookies)
		{
			if(cookie.getName().equals("simulator"))
			{
				if(cookie.getValue().equals("yes"))
					return true;
			}
			
		}
		return false;
	}

		
	/**
	 * Returns the user for specified username
	 * @param a_userName String
	 * @return User
	 */
	public User getSystemUser(String a_userName)
	{
		User result = null;
		for (User user : getUsers())
		{
			if (user.getName().equals(a_userName))
			{
				result = user;
				break;
			}
		}
		return result;
	}

	public User getAdmin()
	{
		User result = null;
		for (User user : getUsers())
		{
			if (user.getType().equals(Constants.IT_USER_TYPE))
			{
				result = user;
				break;
			}
		}
		return result;
	}

	/**
	 * This will return the default Applications available in applications.xml file.
	 * @return Applications
	 */
	public Vector<Application> getDefaultApplications()
	{
		Vector<Application> result = new Vector<Application> ();
		result.addAll(m_defaultUser.getApplications());
		return result;
	}

	/**
	 * This will return the default enabled Applications available in applications.xml file.
	 * @return Applications
	 */
	public Vector<Application> getDefaultEnabledApplications()
	{
		Vector<Application> result = new Vector<Application> ();
		Vector<Application> apps = getDefaultApplications();
		for (Application app : apps)
		{
			if (app.isEnabled())
			{
				result.add(app);
			}
		}
		return result;
	}

	/**
	 * This will return the Users available in users.xml file.
	 * @return Users
	 */
	public Vector<User> getUsers()
	{
		return m_users;
	}

	/**
	 * This method forwards the request to a specified JSP page.
	 * @param a_request HttpServletRequest
	 * @param a_response HttpServletResponse
	 * @param a_page String
	 */
	public void forward(HttpServletRequest a_request,
						HttpServletResponse a_response, String a_page)
	{
		//Cookie cookie = new Cookie("username", "password");
		//cookie.setMaxAge(99);
		try
		{
			RequestDispatcher rd = a_request.getRequestDispatcher(a_page);
			rd.forward(a_request, a_response);
		}
		catch (IOException ex)
		{
			logger.log(Level.SEVERE, ex.getMessage(), ex);
		}
		catch (ServletException ex)
		{
			logger.log(Level.SEVERE, ex.getMessage(), ex);
		}
	}

	/**
	 * Deletes the user from cache and from XML.
	 * @param a_userName String
	 */
	public void deleteUser(String a_userName)
	{
		// REMOVE FROM LOGGED IN USERS
		m_systemUsers.remove(a_userName);
		
		String userAddress = getUserAddress(a_userName);
		if(userAddress!=null)
		{
			m_loggedInUsersCache.remove(userAddress);
		}
		
		Iterator userIt = m_users.iterator();
		while (userIt.hasNext())
		{
			User user = (User) userIt.next();
			if (user.getName().equals(a_userName))
			{
				// DELETE USER AND PERSIST USERS
				userIt.remove();				
				break;
			}
		}
		try
		{
			String usersFilePath = getXmlDirPath() + Constants.USERS_XML_FILE;
			PersistUsers.persist(m_users, usersFilePath);
			String userFilePath = getXmlDirPath() + a_userName + ".xml";
			// DELETE USER SPECIFIC XML FILE
			File file = new File(userFilePath);
			if (file.exists())
			{
				file.delete();
			}
		}
		catch (IOException ex)
		{
			logger.log(Level.SEVERE, ex.getMessage(), ex);
		}
	}
	
	private String getUserAddress(String userName)
	{
		for(String remoteAddress:m_loggedInUsersCache.keySet())
		{
			
			UserProfile profile = m_loggedInUsersCache.get(remoteAddress);
			if(profile.getUser().getName().equals(userName))
				return remoteAddress;
			
		}
		return null;
	}

	/**
	 * This method will load the applications defined in applications.xml file for ADMIN
	 */
	private void loadDefaultUser()
	{
		logger.log(Level.INFO, "loading default Applications defined by ADMIN");
		UserXmlParser parser = new UserXmlParser();
		String appsXmlFile = getXmlDirPath() + Constants.APPLICATIONS_XML_FILE;
		m_defaultUser = parser.parseUserXML(appsXmlFile);
		User admin = getAdmin();
		m_defaultUser.setUser(admin);
		registerApplicationItemsWithRefreshManager(m_defaultUser);
	}

	/**
	 * This method will load the users defined in users.xml file
	 */
	private void loadUsers()
	{
		logger.log(Level.INFO, "Initializing Users");
		UsersXmlParser parser = new UsersXmlParser();
		String usersXmlFile = getXmlDirPath() + Constants.USERS_XML_FILE;
		m_users = parser.parseUserXML(usersXmlFile);
	}

	/**
	 * Created stocks and location directories inside the images directory under sample applications deployed path.
	 */
	private void createDirectories()
	{
		// CREATING STOCKS DIRECTORY
		String stocksBmpPath = getApplicationPath() + Constants.STOCKS_BMP_PATH;
		File stockFile = new File(stocksBmpPath);
		if (!stockFile.exists())
		{
			stockFile.mkdirs();
		}
		// CREATING LOCATIONS DIRECTORY
		String locsBmpPath = getApplicationPath() + Constants.WEATHER_BMP_PATH;
		File locFile = new File(locsBmpPath);
		if (!locFile.exists())
		{
			locFile.mkdirs();
		}
	}

	private void addLogHandler()
	{
		try
		{
			logger.setUseParentHandlers(false);
			//FileHandler fh = new FileHandler(m_appRelativePath + "sampleapps.log");
			FileHandler fh = new FileHandler(m_appRelativePath +
											 "%usampleapps%g.log",
											 50000000
											 /*2MB FILE SIZE*/
											 ,
											 20
											 /* COUNT FOR LOG FILE RECYCLING*/
											 ,
											 false
				/*New file everytime application is stopped and started*/);

			fh.setFormatter(new SimpleFormatter());
			logger.addHandler(fh);
		}
		catch (SecurityException ex)
		{
		}
		catch (IOException ex)
		{
		}
	}
}
