package com.polycom.sampleapps.servlet.common.data;

import java.io.IOException;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Vector;

import com.polycom.sampleapps.servlet.common.ApplicationContext;
import com.polycom.sampleapps.servlet.common.Constants;
import com.polycom.sampleapps.servlet.common.persistance.PersistApplications;
import com.polycom.sampleapps.servlet.common.persistance.PersistUser;
import com.polycom.sampleapps.servlet.exception.LocationAlreadyExistException;
import com.polycom.sampleapps.servlet.exception.RssFeedAlreadyExistException;
import com.polycom.sampleapps.servlet.exception.StockAlreadyExistException;

public class UserProfile
{
	private User m_user;
	private Vector<Application> m_apps;

	public UserProfile()
	{
	}

	public String getName()
	{
		return m_user.getName();
	}

	public String getPassword()
	{
		return m_user.getPassword();
	}

	public String getType()
	{
		return m_user.getType();
	}

	public void setPassword(String a_password)
	{
		m_user.setPassword(a_password);
	}

	public void setApplications(Vector<Application> a_apps)
	{
		m_apps = a_apps;
	}

	public Vector<Application> getApplications()
	{
		return m_apps;
	}

	public void setUser(User a_user)
	{
		m_user = a_user;
	}

	/**
	 * Returns User
	 * @return User
	 */
	public User getUser()
	{
		return m_user;
	}

	/**
	 * Returns this user is of type IT_USER or END_USER
	 * @return boolean
	 */
	public boolean isITUser()
	{
		return m_user.isITUser();
	}

	/**
	 * Returns enabled applications
	 * @return Vector
	 */
	public Vector<Application> getEnabledApplications()
	{
		Vector<Application> result = new Vector<Application> ();
		for (Application app : m_apps)
		{
			if (app.isEnabled())
			{
				result.add(app);
			}
		}
		return result;
	}

	/**
	 * Enables specified applications
	 * @param a_appIds List
	 */
	public void enableApplications(List<String> a_appIds)
	{
		for (Application app : m_apps)
		{
			if (a_appIds.contains(app.getApplicationId()))
			{
				app.setIsEnabled(true);
			}
			else
			{
				app.setIsEnabled(false);
			}
		}
	}

	/**
	 * Updates refresh periods for specified applications
	 * @param a_appIds Map
	 */
	public void updateRefreshPeriod(Map<String, String> a_appIds)
	{
		for (Application app : m_apps)
		{
			if (a_appIds.containsKey(app.getApplicationId()))
			{
				String refreshPeriod = a_appIds.get(app.getApplicationId());
				app.setRefreshPeriod(refreshPeriod);
			}
		}
	}

	/**
	 * Returns Stock application
	 * @return Application
	 */
	public Application getStockApplication()
	{
		Application result = null;
		for (Application app : m_apps)
		{
			if (app.getApplicationId().equals(Application.STOCK_APPLICATION_ID))
			{
				result = app;
				break;
			}
		}
		return result;
	}

	/**
	 * Returns configured stock symbols
	 * @return Vector
	 */
	public Vector<Item> getStocks()
	{
		Vector<Item> result = null;
		Application app = getStockApplication();
		result = app.getItems();
		return result;
	}

	/**
	 * Returns Weather application
	 * @return Application
	 */
	public Application getWeatherApplication()
	{
		Application result = null;
		for (Application app : m_apps)
		{
			if (app.getApplicationId().equals(Application.
											  WEATHER_APPLICATION_ID))
			{
				result = app;
				break;
			}
		}
		return result;
	}

	/**
	 * Returns configured weather locations
	 * @return Vector
	 */
	public Vector<Item> getWeatherLocations()
	{
		Vector<Item> result = null;
		Application app = getWeatherApplication();
		result = app.getItems();
		return result;
	}

	/**
	 * Returns the News application
	 * @return Application
	 */
	public Application getNewsApplication()
	{
		Application result = null;
		for (Application app : m_apps)
		{
			if (app.getApplicationId().equals(Application.NEWS_APPLICATION_ID))
			{
				result = app;
				break;
			}
		}
		return result;
	}

	/**
	 * Returns the configures RSS feed items
	 * @return Vector
	 */
	public Vector<Item> getNews()
	{
		Vector<Item> result = null;
		Application app = getNewsApplication();
		result = app.getItems();
		return result;
	}

	/**
	 * Returns configured thought of the items.
	 * @return Vector
	 */
	public Vector<Item> getThoughtItems()
	{
		Vector<Item> result = null;
		for (Application app : m_apps)
		{
			if (app.getApplicationId().equals(Application.
											  THOUGHT_APPLICATION_ID))
			{
				result = app.getItems();
				break;
			}
		}
		return result;
	}

	/**
	 * Returns the thought of the day application
	 * @return Application
	 */
	public Application getThoughtApplication()
	{
		Application result = null;
		for (Application app : m_apps)
		{
			if (app.getApplicationId().equals(Application.
											  THOUGHT_APPLICATION_ID))
			{
				result = app;
				break;
			}
		}
		return result;
	}

	/**
	 * Adds new RSS feed at first place in the previoulsy configured RSS feeds.
	 * If the RSS feeds are exceeding the maximum allowable limit,
	 * the last item from configured RSS feeds will be deleted.
	 * @param a_rssName String
	 * @param a_url String
	 * @throws RssFeedAlreadyExistException
	 */
	public void addRSS(String a_rssName, String a_url) throws
		RssFeedAlreadyExistException
	{
		boolean isRssExist = false;
		for (Item rss : getNews())
		{
			if (rss.getName().equals(a_rssName))
			{
				isRssExist = true;
				throw new RssFeedAlreadyExistException(a_rssName +
					" already exists.");
			}
		}
		if (!isRssExist)
		{
			Item rss = new Item(a_rssName, a_url, "true");
			getNews().add(0, rss);
			for (Application app : m_apps)
			{
				if (app.getApplicationId().equals("3"))
				{
					int size = getNews().size();
					int maxAllowed = Integer.parseInt(app.getMaxAllowed());
					if (size > maxAllowed)
					{
						getNews().remove(size - 1);
					}
					break;
				}
			}
		}
	}

	/**
	 * Adds new stock symbol at first place in the previoulsy configured stock symbols.
	 * If the stock symbols are exceeding the maximum allowable limit,
	 * the last item from configured stock symbols will be deleted
	 * @param a_stockName String
	 * @throws StockAlreadyExistException
	 */
	public void addStock(String a_stockName) throws StockAlreadyExistException
	{
		Application app = getStockApplication();
		boolean isStockExist = false;
		for (Item stock : app.getItems())
		{
			if (stock.getName().equalsIgnoreCase(a_stockName))
			{
				isStockExist = true;
				throw new StockAlreadyExistException(a_stockName +
					" already exists.");
			}
		}
		if (!isStockExist)
		{
			Item stock = new Item(a_stockName, a_stockName, "true");
			app.getItems().add(0, stock);
			int size = getStocks().size();
			int maxAllowed = Integer.parseInt(app.getMaxAllowed());
			if (size > maxAllowed)
			{
				app.getItems().remove(size - 1);
			}
		}
	}

	/**
	 * Adds new weather location at first place in the previoulsy configured locations.
	 * If the weather locations are exceeding the maximum allowable limit,
	 * the last item from configured locations will be deleted
	 * @param a_locName String
	 * @throws LocationAlreadyExistException
	 */
	public void addLocation(String a_locName) throws
		LocationAlreadyExistException
	{
		Application app = getWeatherApplication();
		boolean isLocationExist = false;
		for (Item location : app.getItems())
		{
			if (location.getName().equalsIgnoreCase(a_locName))
			{
				isLocationExist = true;
				throw new LocationAlreadyExistException(a_locName +
					" already exists.");
			}
		}
		if (!isLocationExist)
		{
			Item location = new Item(a_locName, "/" + a_locName,
									 "true");
			app.getItems().add(0, location);
			int size = getWeatherLocations().size();
			int maxAllowed = Integer.parseInt(app.getMaxAllowed());
			if (size > maxAllowed)
			{
				app.getItems().remove(size - 1);
			}
		}
	}

	/**
	 * Deletes selected stock symbol from home page.
	 * @param a_locName String
	 */
	public void deleteStockFromHomePage(String a_stockName)
	{
		for (Item item : getStocks())
		{
			if (item.getName().equals(a_stockName))
			{
				item.addToHomePage(false);
				break;
			}
		}
	}

	/**
	 * Addss selected stock symbol to home page.
	 * @param a_locName String
	 */
	public void addStockToHomePage(String a_stockName)
	{
		for (Item item : getStocks())
		{
			if (item.getName().equals(a_stockName))
			{
				item.addToHomePage(true);
				break;
			}
		}
	}

	/**
	 * Deletes selected stock symbol from configured stock symbols.
	 * @param a_locName String
	 */
	public void deleteStock(String a_stockName)
	{
		Iterator stockIt = getStocks().iterator();
		while (stockIt.hasNext())
		{
			Item item = (Item) stockIt.next();
			if (item.getName().equals(a_stockName))
			{
				stockIt.remove();
				break;
			}
		}
	}

	/**
	 * Deletes selected stock from home page.
	 * @param a_locName String
	 */
	public void deleteRssFromHomePage(String a_rssName)
	{
		for (Item item : getNews())
		{
			if (item.getName().equals(a_rssName))
			{
				item.addToHomePage(false);
				break;
			}
		}
	}

	/**
	 * Adds selected RSS feed to home page.
	 * @param a_locName String
	 */
	public void addRssToHomePage(String a_rssName)
	{
		for (Item item : getNews())
		{
			if (item.getName().equals(a_rssName))
			{
				item.addToHomePage(true);
				break;
			}
		}
	}

	/**
	 * Deletes selected RSS feed from configured RSS feeds
	 * @param a_locName String
	 */
	public void deleteRss(String a_rssName)
	{
		Iterator rssIt = getNews().iterator();
		while (rssIt.hasNext())
		{
			Item item = (Item) rssIt.next();
			if (item.getName().equals(a_rssName))
			{
				rssIt.remove();
				break;
			}
		}
	}

	/**
	 * Deletes selected weather location from home page.
	 * @param a_locName String
	 */
	public void deleteLocationFromHomePage(String a_locName)
	{
		for (Item item : getWeatherLocations())
		{
			if (item.getName().equals(a_locName))
			{
				item.addToHomePage(false);
				break;
			}
		}
	}

	/**
	 * Adds select weather location to home page
	 * @param a_locName String
	 */
	public void addLocationToHomePage(String a_locName)
	{
		for (Item item : getWeatherLocations())
		{
			if (item.getName().equals(a_locName))
			{
				item.addToHomePage(true);
				break;
			}
		}
	}

	/**
	 * Deletes weather location from configured locations
	 * @param a_locName String
	 */
	public void deleteLocation(String a_locName)
	{
		Iterator locIt = getWeatherLocations().iterator();
		while (locIt.hasNext())
		{
			Item item = (Item) locIt.next();
			if (item.getName().equals(a_locName))
			{
				locIt.remove();
				break;
			}
		}
	}

	/**
	 * Delete Thought application from Home page
	 */
	public void deleteThoughtFromHomePage()
	{
		getThoughtApplication().setIsEnabled(false);
		for (Item item : getThoughtItems())
		{
			item.addToHomePage(false);
		}
	}

	/**
	 * Adds Thought application to home page
	 */
	public void addThoughtToHomePage()
	{
		getThoughtApplication().setIsEnabled(true);
		for (Item item : getThoughtItems())
		{
			item.addToHomePage(true);
		}
	}

	/**
	 * Persists user profile to <USERNAME>.xml file
	 * @param a_appContext ApplicationContext
	 * @throws IOException
	 */
	public void persistUserProfile(ApplicationContext a_appContext) throws IOException
	{
		try
		{
			String userFilePath = "";
			if (m_user.isITUser())
			{
				userFilePath = a_appContext.getXmlDirPath() +
					Constants.APPLICATIONS_XML_FILE;
				PersistApplications.persist(this, userFilePath);
			}
			else
			{
				userFilePath = a_appContext.getXmlDirPath() + getName() +
					".xml";
				PersistUser.persist(this, userFilePath);
			}
		}
		catch (IOException ex)
		{
			throw ex;
		}
	}

	/**
	 * Returns XML string in specified format.
	 * @return String
	 */
	public String getXmlString()
	{
		String output = "";
		//<user name="e" password="p" type="EndUser">
		output += "<user-profile>";
		for (Application app : m_apps)
		{
			output += app.getXmlString();
		}
		output += "</user-profile>";
		return output;
	}

	public String toString()
	{
		return getName() + " : " + m_apps;
	}
}
