package com.polycom.sampleapps.servlet.news;

import java.util.HashMap;
import java.util.Vector;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;

import com.polycom.sampleapps.servlet.common.ApplicationContext;
import com.polycom.sampleapps.servlet.common.CheckURL;
import com.polycom.sampleapps.servlet.common.data.Channel;
import com.polycom.sampleapps.servlet.common.parsers.RssXmlParser;

/**
*
* <p>Title: NewsManager</p>
*
* <p>Description: Holds the list of rss feeds configured for the users and caches the news content for given News Rss feeds.
* Provides methods to get the relavant information from cache. </p>
*
* <p>Copyright:* Copyright (c) 2007 Polycom Canada Ltd.</p>
*
* <p>Company: Polycom, Inc</p>
*
* @author M Sandeep
* @version 1.0
*/
public class NewsManager {

	private ApplicationContext m_appContext;

	private Vector<String> currentActiveNewsVec; // to hold current active

	// news rss feeds - content map
	private ConcurrentMap<String, Channel> m_newsChannelTable; // Key--> News RSS , Value--> News content
	
	//news rss feeds - last access time of feed
	private HashMap<String, Long> lastAccessTimeMap;

	public NewsManager(ApplicationContext appContext) {
		m_appContext = appContext;
		currentActiveNewsVec = new Vector<String>();
		
		m_newsChannelTable = new ConcurrentHashMap<String, Channel>();
		lastAccessTimeMap=new HashMap<String, Long>();
		initNewsDataProcessor();
	}

	/**
	 * Initializes the news processor which will fetch new data for configured news for every 15 min
	 *
	 */
	private void initNewsDataProcessor() {
		NewsDataProcessor newsCreator = new NewsDataProcessor(this);
		newsCreator.start();
	}

	/**
	 * Registers a RSS news feed provider url with the cache and invokes a thread to read the content from given url.
	 * @param rssFeed
	 */
	public void add(String rssFeed) {
		if (!currentActiveNewsVec.contains(rssFeed)) {
			currentActiveNewsVec.add(rssFeed);
		}
	}

	/**
	 * Checks if given rss feed is existing in current list of feeds
	 * @param rssFeed
	 * @return
	 */
	public boolean isNewsExists(String rssFeed) {
		return currentActiveNewsVec.contains(rssFeed);
	}

	/**
	 * Returns the result, RSS News feed is processed inside a thread and the result Channel object is cached inside data processor
	 * If returns null, RSS News feed is not yet processed.
	 * @param rssFeed
	 * @return
	 */
	public Channel getNewsInfo(String rssFeed) {
		Channel result = m_newsChannelTable.get(rssFeed);
		lastAccessTimeMap.put(rssFeed, System.currentTimeMillis());		
		return result;
	}

	/**
	 * Returns the RSS Feed and news channel object map
	 * @return
	 */
	public ConcurrentMap<String, Channel> getNewsChannelTable() {
		return m_newsChannelTable;
	}

	/**
	 * Deletes RSS News feed from current news list
	 * @param rssFeed
	 */
	public void delete(String rssFeed) {
		if(currentActiveNewsVec.contains(rssFeed))
			currentActiveNewsVec.remove(rssFeed);
		if(m_newsChannelTable.containsKey(rssFeed))
			m_newsChannelTable.remove(rssFeed);
	}

	/**
	 * Returns currently active RSS feed url's
	 * @return
	 */
	public Vector<String> getActiveNews() {
		return currentActiveNewsVec;
	}

	/**
	 * Used to check whether the given rssFeed is valid or not(Used for adding new RSS feed)
	 * @param rssFeed
	 * @return
	 */
	public String isValidRssFeed(String rssFeed) {
		String result = "";
		int noOfRetries = 0;
		try {
			while (true) {
				CheckURL m_CheckURL = new CheckURL(rssFeed, false);
				Thread channelThread = new Thread(m_CheckURL);
				channelThread.run();
				String content = m_CheckURL.getResult().toString();
				m_CheckURL=null;
				
				RssXmlParser parser = new RssXmlParser();
				Channel channel = parser.parseRssXML(content);
				if (channel != null) {					
					m_newsChannelTable.put(rssFeed, channel);
					add(rssFeed);
					break;
				}
				if (noOfRetries == 3) {
					result = "Unable add news/Invalid RSS feed. Please try after some time.";
					break;
				}
				noOfRetries++;
			}
		} catch (Exception e) {
			// TODO Auto-generated catch block
			result = "Unable add news. Please try after some time.";
		}
		return result;
	}


	/**
	 * Returns map consisting of RSS feed url and the last access time of the feed
	 * @return
	 */
	public HashMap<String, Long> getNewsAccessMap() {
		return lastAccessTimeMap;
	}
}
