package com.polycom.sampleapps.servlet.stocks;

import java.io.File;
import java.util.HashMap;
import java.util.Vector;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;

import com.polycom.sampleapps.servlet.common.ApplicationContext;
import com.polycom.sampleapps.servlet.common.CheckURL;
import com.polycom.sampleapps.servlet.common.Constants;


/**
*
* <p>Title: StockManager</p>
*
* <p>Description: Holds the list of stock symbols configured for the users and caches the stockinfo objects for given stock symbol.
* Provides methods to get the relavant information from cache. </p>
*
* <p>Copyright:* Copyright (c) 2007 Polycom Canada Ltd.</p>
*
* <p>Company: Polycom, Inc</p>
*
* @author M Sandeep
* @version 1.0
*/
public class StockManager {
	
	private ApplicationContext m_appContext;

	private Vector<String> currentActiveSymbolsVec; // to hold current active
													// symbols.

	private ConcurrentMap<String, StockInfo> stockInfoHt; // key -> symbol & value-> StockInfo
	
	private HashMap<String, Long> stockAccessMap;// key -> symbol & value-> Last access time

	public StockManager(ApplicationContext appContext) {
		m_appContext = appContext;
		currentActiveSymbolsVec = new Vector<String>();
		stockInfoHt = new ConcurrentHashMap<String, StockInfo>();
		stockAccessMap=new HashMap<String, Long>();
		initStockDataProcessor();
	}

	public String getStockBmpPath() {
		return m_appContext.getStocksDirPath();
	}

	/**
	 * Checks if given stock symbol is existing in current list of symbols
	 * @param symbol
	 * @return
	 */
	public boolean isSymbolExists(String symbol) {
		return currentActiveSymbolsVec.contains(symbol);
	}

	/**
	 * Initializes the stocks processor which will fetch new data for configured stocks for every 15 min
	 *
	 */
	public void initStockDataProcessor() {
		StockDataProcessor sdp = new StockDataProcessor(this);
		sdp.start();
	}

	/**
	 * Registers stock symbol with the cache 
	 * @param symbol
	 */
	public void add(String symbol) {
		if (!currentActiveSymbolsVec.contains(symbol)) {
			currentActiveSymbolsVec.add(symbol);			
		}
	}

	/**
	 * Return the location of bmp file for given stock symbol.
	 * @param symbol
	 * @return
	 */
	public String getBmpFileName(String symbol) {
		String result = null;
		StockInfo info = stockInfoHt.get(symbol);
		if (info != null) {
			result = info.getBmpFileName();
		}
		stockAccessMap.put(symbol,System.currentTimeMillis());
		return result;
	}

	/**
	 * Returns the result, Stock symbol is processed inside a thread and the result (stockinfo object) is cached inside data processor
	 * If returns null, stock symbol is not yet processed.
	 * @param rssFeed
	 * @return
	 */
	public StockInfo getStockInfo(String symbol) {
		StockInfo result = stockInfoHt.get(symbol);
		stockAccessMap.put(symbol,System.currentTimeMillis());
		return result;
	}

	/**
	 * Deletes Stock symbol from current news list
	 * @param symbol
	 */
	public void delete(String symbol) {
		if(currentActiveSymbolsVec.contains(symbol))
			currentActiveSymbolsVec.remove(symbol);
		if(stockInfoHt.containsKey(symbol))
			stockInfoHt.remove(symbol);
	}

	/**
	 * Returns currently configured stocks list
	 * @return
	 */
	public Vector<String> getActiveSymbols() {
		return currentActiveSymbolsVec;
	}

	/**
	 * Returns map of stock and stockinfo which will used to display stock information in phone 
	 * @return
	 */
	public ConcurrentMap<String, StockInfo> getStockInfoHt() {
		return stockInfoHt;
	}
	
	/**
	 * Returns map of stock and last access time which will be used to update the stock informtion using NHTTP client
	 * @return
	 */
	public HashMap<String, Long> getStockAccessMap() {
		return stockAccessMap;
	}


	/**
	 * Used to check whether the given symbol is valid or not(Used for adding new symbol)
	 * @param rssFeed
	 * @return
	 */
	public String isValidStockSymbol(String symbol) {
		String result = "";
		String quoteURL = "http://finance.yahoo.com/q?s=" + symbol;
		String chartURL = "http://ichart.yahoo.com/t?s=" + symbol;
		int noOfRetries = 0;
		try {
			while (true) {
				CheckURL m_CheckURL = new CheckURL(quoteURL, false);
				Thread quoteThread = new Thread(m_CheckURL);
				quoteThread.run();
				String content = m_CheckURL.getResult().toString();
				m_CheckURL=null;
				
				CheckURL m_CheckURL2 = new CheckURL(chartURL, true);
				Thread chartThread = new Thread(m_CheckURL2);
				chartThread.run();				
				byte[] chartBytes = (byte[]) m_CheckURL2.getResult();
				m_CheckURL2=null;
				
				StockQuoteHtmlParser parser = new StockQuoteHtmlParser();
				parser.readQuote(content);
				if (parser.getLastTrade().length() > 0
						&& parser.getChange().length() > 0
						&& parser.getVolume().length() > 0
						&& chartBytes != null && chartBytes.length > 0) {					
					StockInfo info = StockBmpConverter.createBMP(symbol, content, chartBytes, getStockBmpPath());					
					if(info!=null)
					{
						stockInfoHt.put(symbol, info);
						currentActiveSymbolsVec.add(symbol);
						break;
					}
				}

				if (noOfRetries == 3) {
					result = "Unable add stock/Invalid stock symbol. Please try after some time.";
					break;
				}
				noOfRetries++;
			}
		} catch (Exception e) {
			// TODO Auto-generated catch block
			result = "Unable add weather. Please try after some time.";
		}

		return result;
	}

	/**
	 * Deletes the unused images which are not accesed in 20 min
	 *
	 */
	public void deleteItems() {
		
		long currTime = System.currentTimeMillis();
		
		// Get All Files
		File file = new File(getStockBmpPath());
		File[] files = file.listFiles();
		for (File f : files) {
			long lastModified = f.lastModified();			
			if ((currTime - lastModified) > Constants.DEFAULT_IMAGES_CLEANUP_PERIOD) {
				f.delete();
			}
		}
	}

	public static void main(String[] args) {
		//StockManager.isValidStockSymbol("DELL");
	}
}
