package com.polycom.sampleapps.servlet.stocks;

import java.awt.image.BufferedImage;
import java.io.IOException;
import java.io.File;
import java.util.Date;
import java.util.Hashtable;
import java.util.logging.Level;

import javax.swing.text.BadLocationException;

import com.polycom.sampleapps.servlet.common.ApplicationContext;
import com.polycom.sampleapps.servlet.common.Constants;
import com.polycom.sampleapps.servlet.common.ContentReader;
import com.polycom.sampleapps.servlet.common.Reader;
import com.polycom.sampleapps.servlet.common.RefreshManager;

public class StockQuoteReader implements Reader
{
	public static int STOCK_COUNTER = 1;

	private static final String STOCK_QUOTE_CONTENT_PROVIDER_URL =
		"http://finance.yahoo.com/q?s=";
	private static final String STOCK_CHART_CONTENT_PROVIDER_URL =
		"http://ichart.yahoo.com/t?s=";

	private static StockQuoteReader m_self;
	private RefreshManager m_manager;
	private Hashtable<String, StockInfo> m_stockQuoteInfoTable;
	private Hashtable<String, Long> m_stockQuoteAccessedTimeTable;

	private StockQuoteReader(RefreshManager a_manager)
	{
		m_manager = a_manager;
		m_stockQuoteAccessedTimeTable = new Hashtable<String, Long> ();
		m_stockQuoteInfoTable = new Hashtable<String, StockInfo> ();
	}

	public static StockQuoteReader getInstance(RefreshManager a_manager)
	{
		if (m_self == null)
		{
			m_self = new StockQuoteReader(a_manager);
		}
		return m_self;
	}

	/**
	 * Checks for last accssed time, if it exceeds beyond the specified accessed time, then unregister this stock quote from cache.
	 * If it is with in specified access time limit, reads stock quote and chart content from content provider site for specified stock symbol.
	 * After reading, parses the HTML content and create stock chart and stock info object and then caches it.
	 * @param a_itemName String
	 * @return Object
	 */
	public Object read(String a_itemName)
	{
		//ApplicationContext.logger.log(Level.INFO, a_itemName + " READING STARTED...........");
		StockInfo stockInfo = null;
		while (true)
		{
			StockQuoteHtmlParser quoteHtmlParser = null;
			BufferedImage arrowImage = null;
			BufferedImage stockImage = null;
			BufferedImage targetImage = null;
			try
			{
				// CHECKING FOR LAST ACCESSED TIME, IF IT IS MORE THAN 5 MINS, THEN UNREGISTER THIS ITEM
				Object obj = m_stockQuoteAccessedTimeTable.get(a_itemName);
				if (obj != null)
				{
					long lastAccessedTime = ( (Long) obj).longValue();
					long currentTime = System.currentTimeMillis();
					long meanTime = currentTime - lastAccessedTime;
					if (meanTime >= Constants.UNREGISTER_ITEM_FROM_CACHE_TIME)
					{
						m_stockQuoteAccessedTimeTable.remove(a_itemName);
						m_manager.unregister(a_itemName);

						return stockInfo;
					}
				}
				// READ AND PARSE STOCK QUOTE INFORMATION USING DEFINED CONTENT PROVIDER URL
				String quoteUrl = STOCK_QUOTE_CONTENT_PROVIDER_URL + a_itemName;
				quoteHtmlParser = new StockQuoteHtmlParser();
				//quoteHtmlParser.readQuote(quoteUrl, "");
				if (quoteHtmlParser.getLastTrade().length() > 0)
				{
					// NOW READ BYTE CONTENT OF CHANGE IMAGE
					byte[] chartBytes = null;
					byte[] arrowBytes = null;
					String arrowUrl = quoteHtmlParser.getChangeImage().trim();
					if (arrowUrl != null && arrowUrl.length() > 0)
					{
						arrowBytes = ContentReader.getByteContent(arrowUrl);
					}
					// NOW READ BYTE CONTENT OF CHART IMAGE
					String chartUrl = STOCK_CHART_CONTENT_PROVIDER_URL +
						a_itemName;
					chartBytes = ContentReader.getByteContent(chartUrl);
					// CREATE STOCK INFO AND CACHE IT AGAINST ITEM
					if(chartBytes!=null && chartBytes.length>0)
					{
						// 1. CREATE BMP FILE WITH STOCK SYMBOL NAME(<STOCK_SYMBOL_COUNTER>)
						if(STOCK_COUNTER > 30000)
						{
							STOCK_COUNTER = 1;
						}
						String bmpFileName = a_itemName+"_"+(STOCK_COUNTER++) +".bmp";
						String filePath = m_manager.getBmpFilePath()+File.separator+bmpFileName;
						stockInfo = new StockInfo(a_itemName, quoteHtmlParser.
											  getLastTrade(),
											  quoteHtmlParser.getVolume(),
											  quoteHtmlParser.getChange(),
											  arrowUrl, bmpFileName);
											  //arrowBytes, chartBytes);

						//StockBmpConverter.convertToBmp(stockInfo, chartBytes, arrowBytes, filePath);

						// 2. PUT FILE NAME IN StockInfo.
						File file = new File(filePath);
						if (file.exists())
						{

							m_stockQuoteInfoTable.put(a_itemName, stockInfo);					
							break;
						}
					}else
					{
						m_stockQuoteAccessedTimeTable.remove(a_itemName);
						m_manager.unregister(a_itemName);
						break;
					}
				}
			}
			catch (IOException ex)
			{
				ApplicationContext.logger.log(Level.INFO, "Could not found stock");
			}
			catch (Exception ex)
			{
				ApplicationContext.logger.log(Level.INFO, "Got exception in stock quote reader-->", ex.getMessage());
			}
			finally
			{
				if (stockImage != null)
				{
					stockImage.flush();
					stockImage = null;
				}
				if (targetImage != null)
				{
					targetImage.flush();
					targetImage = null;
				}
				if (arrowImage != null)
				{
					arrowImage.flush();
					arrowImage = null;
				}
				if (quoteHtmlParser != null)
				{
					try
					{
						quoteHtmlParser.flush();
					}
					catch (BadLocationException ex1)
					{
					}
					quoteHtmlParser = null;
				}
				break;
			}
		}
		return stockInfo;
	}

	/**
	 * Updates the last accessed time and returns the latest Stock quote information.
	 * @param a_itemName String
	 * @return Object
	 */
	public Object getResult(String a_itemName)
	{
		Object result = m_stockQuoteInfoTable.get(a_itemName);
		m_stockQuoteAccessedTimeTable.put(a_itemName, System.currentTimeMillis());
		return result;
	}

	public void deleteCache(String a_itemName)
	{
		m_stockQuoteInfoTable.remove(a_itemName);
		m_stockQuoteAccessedTimeTable.remove(a_itemName);
	}
}
