package com.polycom.sampleapps.servlet.weather;

import java.io.File;
import java.util.HashMap;
import java.util.Vector;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;

import com.polycom.sampleapps.servlet.common.ApplicationContext;
import com.polycom.sampleapps.servlet.common.CheckURL;
import com.polycom.sampleapps.servlet.common.Constants;

/**
*
* <p>Title: WeatherManager</p>
*
* <p>Description: Holds the list of weather items configured for the users and caches the weatherinfo objects for given weather item.
* Provides methods to get the relavant information from cache. </p>
*
* <p>Copyright:* Copyright (c) 2007 Polycom Canada Ltd.</p>
*
* <p>Company: Polycom, Inc</p>
*
* @author M Sandeep
* @version 1.0
*/
public class WeatherManager {

	private ApplicationContext m_appContext;

	private Vector<String> currentActiveItemsVec; // to hold current active

	// symbols.
	private ConcurrentMap<String, WeatherInfo> weatherInfoHt; // key -> symbol &

	// value ->
	// WeatherInfo
	
	private HashMap<String, Long> weatherAccessMap;

	public WeatherManager(ApplicationContext appContext) {
		m_appContext = appContext;
		currentActiveItemsVec = new Vector<String>();
		weatherInfoHt = new ConcurrentHashMap<String, WeatherInfo>();
		weatherAccessMap=new HashMap<String, Long>();
		initStockDataProcessor();
	}

	/**
	 * Return the location of bmp files to be placed in server
	 * @param symbol
	 * @return
	 */
	public String getlocationsBmpPath() {
		return m_appContext.getLocationsDirPath();
	}

	/**
	 * Checks if given weather item is existing in current list of items
	 * @param symbol
	 * @return
	 */
	public boolean isSymbolExists(String symbol) {
		return currentActiveItemsVec.contains(symbol);
	}

	/**
	 * Initializes the weather processor which will fetch new data for configured items for every 15 min
	 *
	 */
	public void initStockDataProcessor() {
		WeatherDataProcessor sdp = new WeatherDataProcessor(this);
		sdp.start();
	}


	/**
	 * Registers weather symbol with the cache 
	 * @param symbol
	 */
	public void add(String symbol) {
		if (!currentActiveItemsVec.contains(symbol)) {
			currentActiveItemsVec.add(symbol);
		}
	}

	/**
	 * Return the location of bmp file for given weather item.
	 * @param symbol
	 * @return
	 */
	public String getBmpFileName(String symbol) {
		String result = null;
		WeatherInfo info = weatherInfoHt.get(symbol);
		if (info != null) {
			result = info.getBmpFileName();
		}
		weatherAccessMap.put(symbol, System.currentTimeMillis());
		return result;
	}


	/**
	 * Returns the WeatherInfo object, weather item is processed inside a thread and the result (WeatherInfo object) is cached inside data processor
	 * If returns null, weather item  is not yet processed.
	 * @param rssFeed
	 * @return
	 */
	public WeatherInfo getWeatherInfo(String symbol) {
		WeatherInfo result = weatherInfoHt.get(symbol);
		weatherAccessMap.put(symbol, System.currentTimeMillis());
		return result;
	}

	/**
	 * Deletes weather item from current news list
	 * @param symbol
	 */
	public void delete(String symbol) {
		if(currentActiveItemsVec.contains(symbol))
			currentActiveItemsVec.remove(symbol);
		if(weatherInfoHt.containsKey(symbol))
			weatherInfoHt.remove(symbol);
	}

	/**
	 * Return currently configured weather items
	 * @return
	 */
	public Vector<String> getActiveItems() {
		return currentActiveItemsVec;
	}

	public ConcurrentMap<String, WeatherInfo> getWeatherInfoHt() {
		return weatherInfoHt;
	}

	/**
	 * Used to check whether the given item is valid or not(Used for adding new item)
	 * @param rssFeed
	 * @return
	 */
	public String isValidWeatherSymbol(String symbol) {
		String result = "";
		String url = "http://www.weather.com/weather/local/" + symbol;
		int noOfRetries = 0;
		try {
			while (true) {
				CheckURL m_CheckURL = new CheckURL(url, false);
				Thread check = new Thread(m_CheckURL);
				check.run();
				String content = m_CheckURL.getResult().toString();
				m_CheckURL=null;
				
				WeatherHtmlParser parser = new WeatherHtmlParser();
				parser.readWeatherContent(content);
				if (parser.getTemperature() != null) {
					WeatherInfo info = WeatherBMPMaker.createBMP(symbol,
							content, getlocationsBmpPath());					
					if (info != null) {
						weatherInfoHt.put(symbol, info);
						add(symbol);
						break;
					}
				}
				if (noOfRetries == 3) {
					result = "Unable add Weather/Invalid Weather symbol. Please try after some time.";
					break;
				}
				noOfRetries++;
			}
		} catch (Exception e) {
			// TODO Auto-generated catch block
			result = "Unable add weather. Please try after some time.";
		}

		return result;
	}
	
	
	/**
	 * Deletes the unused images which are not accesed in 20 min
	 *
	 */
	public void deleteItems() {
		long currTime = System.currentTimeMillis();
		// Get All Files
		File file = new File(getlocationsBmpPath());
		File[] files = file.listFiles();
		for (File f : files) {
			long lastModified = f.lastModified();			
			if ((currTime - lastModified) > Constants.DEFAULT_IMAGES_CLEANUP_PERIOD) {
				f.delete();
			}
		}
				
	}

	public HashMap<String, Long> getWeatherAccessMap() {
		return weatherAccessMap;
	}

}
