package com.polycom.sampleapps.servlet.weather;

import java.util.Hashtable;
import java.util.Iterator;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.logging.Level;

import com.polycom.sampleapps.servlet.common.ApplicationContext;
import com.polycom.sampleapps.servlet.common.Constants;
import com.polycom.sampleapps.servlet.common.Reader;
import com.polycom.sampleapps.servlet.common.RefreshManager;
import com.polycom.sampleapps.servlet.common.RunnableReader;
import com.polycom.sampleapps.servlet.common.ThreadInfo;
import com.polycom.sampleapps.servlet.exception.ItemNotRegisteredException;

public class WeatherRefreshManager implements RefreshManager
{
	private static int MINIMUM_THREAD_COUNT = 200; // Maximum number of threads in the pool
	private static WeatherRefreshManager m_self;
	private Reader m_weatherReader; // Reader to read the content from content provider site
	private ScheduledThreadPoolExecutor m_execuotorService; // Scheduled executor thread pool
	// Contains the location name as key and ScheduledFuture of periodic excution thread as value.
	private Hashtable<String, ThreadInfo> m_scheduleTable;

	private WeatherRefreshManager()
	{
		m_weatherReader = WeatherReader.getInstance(this);
		m_scheduleTable = new Hashtable<String, ThreadInfo> ();
		m_execuotorService = (ScheduledThreadPoolExecutor) Executors.
			newScheduledThreadPool(MINIMUM_THREAD_COUNT);
	}

	public static synchronized WeatherRefreshManager getInstance()
	{
		if (m_self == null)
		{
			m_self = new WeatherRefreshManager();
		}
		return m_self;
	}

	public String getBmpFilePath()
	{
		return null;
	}

	/**
	 * Registers a location name with the cache and invokes a thread to read the content from content provider url.
	 * @param a_itemName String
	 * @return Object
	 */
	public Object register(String a_itemName)
	{
		// CREATE NEW THREAD TO REFRESH THE WEATHER INFORMATION
		if (!m_scheduleTable.containsKey(a_itemName))
		{
			ApplicationContext.logger.log(Level.INFO,
										  "Creating scheduler for Item : " +
										  a_itemName);
			scheduleReadingWeatherInfo(a_itemName);
		}
		return null;
	}

	/**
	 * If returns the result, weather information is processed inside a thread and cached bmp file and weather info object inside the reader.
	 * If returns null, weather information is not yet processed.
	 * @param a_itemName String
	 * @return Object
	 */
	public Object getResult(String a_itemName) throws
		ItemNotRegisteredException
	{
		Object result = m_weatherReader.getResult(a_itemName);
		if (result == null)
		{
			ThreadInfo thInfo = m_scheduleTable.get(a_itemName);
			if (thInfo != null)
			{
				ScheduledFuture future = thInfo.getFuture();
				if (future != null)
				{
					try
					{
						future.get(Constants.CONTENT_READ_RFRESH_RATE,
								   TimeUnit.MILLISECONDS);
						result = m_weatherReader.getResult(a_itemName);
					}
					catch (TimeoutException ex1)
					{
					}
					catch (ExecutionException ex1)
					{
						unregister(a_itemName);
						register(a_itemName);
					}
					catch (InterruptedException ex1)
					{
						unregister(a_itemName);
						register(a_itemName);
					}
				}
				else
				{
					throw new ItemNotRegisteredException(a_itemName +
						" not registered");
				}
			}
			else
			{
				throw new ItemNotRegisteredException(a_itemName +
					" not registered");
			}
		}
		return result;
	}

	/**
	 * Unregisters the Location name from cache and cancels the refreshing thread.
	 * @param a_itemName String
	 */
	public void unregister(String a_itemName)
	{
		try
		{
			ApplicationContext.logger.log(Level.INFO,
										  "Unregistering Item : " + a_itemName);
			ThreadInfo thInfo = m_scheduleTable.get(a_itemName);
			if ( thInfo != null )
			{
				ScheduledFuture future = thInfo.getFuture();
				if (future != null && !future.isCancelled())
				{
					future.cancel(true);
				}
			}
			m_scheduleTable.remove(a_itemName);
			m_weatherReader.deleteCache(a_itemName);
			m_execuotorService.purge();
		}
		catch (Exception ex)
		{
		}
	}

	/**
	 * Shutdowns/Cancels the tasks that are excuting.
	 */
	public void shutdown()
	{
		m_execuotorService.shutdown();
	}

	public boolean isItemExist(String a_itemName)
	{
		boolean result = false;
		if (m_scheduleTable.containsKey(a_itemName))
		{
			result = true;
		}
		return result;
	}

	public void interruptBlockedThreads()
	{
		Hashtable<String, ThreadInfo> temp = new Hashtable<String,
			ThreadInfo> ();
		temp.putAll(m_scheduleTable);
		Iterator it = temp.keySet().iterator();
		while (it.hasNext())
		{
			String itemName = (String) it.next();
			ThreadInfo thInfo = (ThreadInfo) temp.get(itemName);
			RunnableReader runnable = thInfo.getRunnable();
			if (runnable.releaseFromBlockedState())
			{
				try
				{
					unregister(runnable.getItemName());
					register(runnable.getItemName());
				}
				catch (Exception ex)
				{
				}
			}
		}
	}

	/**
	 * Create and Schedules thread for weather location with given refresh rate and updates weather location information in the cache.
	 * @param a_itemName String
	 */
	private void scheduleReadingWeatherInfo(String a_itemName)
	{
		RunnableReader runReader = new RunnableReader(m_weatherReader,
			a_itemName, Constants.DEFAULT_WEATHER_THREAD_TIMEOUT);
		ScheduledFuture future = m_execuotorService.scheduleWithFixedDelay(
			runReader, 0, Constants.DEFAULT_WEATHER_CONTENT_READ_TIMEOUT,
			TimeUnit.MILLISECONDS);
		ThreadInfo thInfo = new ThreadInfo(runReader, future);
		m_scheduleTable.put(a_itemName, thInfo);
	}
}
